# Laravel Production Readiness Checklist

This checklist outlines steps to ensure the "lapor-mas-wapress" application is ready for a production environment.

## 1. Environment Configuration (`.env`)

*   [ ] **Create `.env` file:** Copy `.env.example` to `.env` on the production server. **Never commit your production `.env` file to version control.**
*   [ ] **`APP_ENV`:** Set `APP_ENV=production`.
*   [ ] **`APP_DEBUG`:** Set `APP_DEBUG=false`.
*   [ ] **`APP_KEY`:** Generate a unique key using `php artisan key:generate` on the production server and copy it to `.env`.
*   [ ] **`APP_URL`:** Set the correct public URL of your application (e.g., `APP_URL=https://yourdomain.com`).
*   [ ] **Database Credentials:** Configure `DB_CONNECTION`, `DB_HOST`, `DB_PORT`, `DB_DATABASE`, `DB_USERNAME`, `DB_PASSWORD` for the production database (e.g., use `mysql` or `pgsql`).
*   [ ] **Cache Driver:** Set `CACHE_STORE` to a production-suitable driver like `redis` or `memcached`. Configure the corresponding connection details (e.g., `REDIS_HOST`, `REDIS_PORT`).
*   [ ] **Queue Driver:** Set `QUEUE_CONNECTION` to a production-suitable driver like `redis` or `sqs`. Configure the corresponding connection details. Ensure a queue worker process is running (`php artisan queue:work --daemon` or using Supervisor/Systemd).
*   [ ] **Session Driver:** Set `SESSION_DRIVER` to `redis`, `memcached`, or `database`. Configure `SESSION_CONNECTION` if needed.
*   [ ] **Mail Driver:** Configure `MAIL_MAILER`, `MAIL_HOST`, `MAIL_PORT`, `MAIL_USERNAME`, `MAIL_PASSWORD`, `MAIL_ENCRYPTION`, `MAIL_FROM_ADDRESS`, `MAIL_FROM_NAME` for sending emails (e.g., using `smtp`, `ses`).
*   [ ] **reCAPTCHA Keys:** Set `RECAPTCHA_SECRET_KEY` (and site key if needed elsewhere). **Consider moving this to `config/services.php`**.
*   [ ] **HTTPS:** If using HTTPS, set `SESSION_SECURE_COOKIE=true`.
*   [ ] **Domain:** Set `SESSION_DOMAIN` to your root domain if needed (e.g., `.yourdomain.com`).

## 2. Security

*   [ ] **HTTPS:** Ensure your web server (Nginx/Apache) is configured to serve the site over HTTPS and redirects HTTP traffic.
*   [ ] **Permissions:** Set correct file permissions. Web server needs write access to `storage/` and `bootstrap/cache/`. Other files should generally not be writable by the web server.
    ```bash
    # Example (adjust user/group as needed)
    sudo chown -R $USER:www-data .
    sudo find storage bootstrap/cache -type d -exec chmod 775 {} \;
    sudo find storage bootstrap/cache -type f -exec chmod 664 {} \;
    ```
*   [ ] **Remove `env()` from Code:** Replace `env('RECAPTCHA_SECRET_KEY')` in `RegistrationController.php` with `config('services.recaptcha.secret')`. Add the following to `config/services.php`:
    ```php
    // config/services.php
    'recaptcha' => [
        'secret' => env('RECAPTCHA_SECRET_KEY'),
        // Add site key if needed
        // 'sitekey' => env('RECAPTCHA_SITE_KEY'),
    ],
    ```
*   [ ] **File Upload Validation:** Review and ensure consistency between manual size checks and validation rules (`max:xxxx`) in `RegistrationController.php`.
*   [ ] **Rate Limiting:** Consider adding rate limiting to sensitive routes like login (`/admin/login`) and registration (`/register`) using Laravel's built-in features.
*   [ ] **Update Dependencies:** Regularly update dependencies (`composer update`, `npm update`) to patch security vulnerabilities. Use `composer audit` and `npm audit` to check.
*   [ ] **Hide Sensitive Files:** Ensure web server configuration prevents direct access to files like `.env`, `storage/logs`, etc.

## 3. Performance

*   [ ] **Configuration Caching:** Run these commands during deployment:
    ```bash
    php artisan config:cache
    php artisan route:cache
    php artisan view:cache
    ```
    *Note: Run `php artisan optimize:clear` if you need to clear these caches.*
*   [ ] **Autoloader Optimization:** Use the optimized autoloader:
    ```bash
    composer install --no-dev --optimize-autoloader
    # or if already installed:
    # composer dump-autoload --optimize
    ```
*   [ ] **Frontend Assets:** Build optimized assets:
    ```bash
    npm ci --production # Install only production dependencies
    npm run build
    ```
*   [ ] **OpCache:** Ensure PHP OpCache is enabled and configured optimally on your server.
*   [ ] **Database Indexing:** Review migrations and add indexes to columns frequently used in `WHERE` clauses (e.g., `registrations.nik`, `registrations.visit_date`, `registrations.visit_time`, `registrations.booking_code`, `transaksi.waktudatang`, `transaksi.id_online`, `transaksi.nik`).
*   [ ] **Queue Workers:** Ensure queue workers are running reliably using a process manager like Supervisor or Systemd to process jobs in the background.
*   [ ] **Choose Performant Drivers:** Use `redis` or `memcached` for Cache and Session if possible. Use `redis` or `sqs` for Queues.

## 4. Deployment

*   [ ] **Version Control:** Use Git and have a clear branching strategy (e.g., main/master for production).
*   [ ] **Deployment Script/CI/CD:** Automate the deployment process using scripts or a CI/CD pipeline (e.g., GitHub Actions, GitLab CI, Jenkins).
*   [ ] **Deployment Steps:**
    1.  Pull latest code from the production branch.
    2.  Install/update Composer dependencies: `composer install --no-dev --optimize-autoloader`
    3.  Install/update NPM dependencies and build assets: `npm ci --production && npm run build`
    4.  Run database migrations: `php artisan migrate --force` (`--force` is required in production).
    5.  Cache configuration: `php artisan config:cache`, `php artisan route:cache`, `php artisan view:cache`.
    6.  Restart queue workers if necessary.
    7.  Clear relevant application caches if needed (`php artisan cache:clear`).
*   [ ] **Zero-Downtime Deployment:** Consider using tools/strategies like Envoyer or symlinking for zero-downtime deployments.

## 5. Logging & Error Handling

*   [ ] **Logging Channel:** Configure `LOG_CHANNEL` in `.env` to `daily`, `stack` (with appropriate sub-channels), or an external service driver (Slack, Papertrail, Sentry).
*   [ ] **Log Level:** Set `LOG_LEVEL` in `.env` appropriately (e.g., `warning`, `error`, `critical`). Avoid `debug` in production unless actively troubleshooting.
*   [ ] **Error Reporting:** Ensure `APP_DEBUG=false` hides detailed error pages from users. Consider using a service like Sentry for error tracking.

## 6. Maintenance & Backups

*   [ ] **Maintenance Mode:** Understand how to use maintenance mode:
    ```bash
    php artisan down
    # --secret="your-secret" allows bypassing
    # --render="errors::503" uses a custom view
    php artisan up
    ```
*   [ ] **Backups:** Implement regular backups for:
    *   Database (e.g., using `mysqldump`, `pg_dump`, or cloud provider tools).
    *   Uploaded files (e.g., the `storage/app/private/ktp_images` directory).
*   [ ] **Scheduled Tasks:** If you add any scheduled tasks in `app/Console/Kernel.php`, ensure the Laravel scheduler is configured to run via cron on the server.
    ```cron
    * * * * * cd /path-to-your-project && php artisan schedule:run >> /dev/null 2>&1
    ```

## 7. Docker (If Using for Production)

*   [ ] **Production Dockerfile:** Create a separate `Dockerfile` (or use multi-stage builds) optimized for production (smaller image, no dev dependencies, correct permissions, optimized asset building).
*   [ ] **Production Compose/Orchestration:** Adapt `docker-compose.yml` or use a production-grade orchestrator (Kubernetes, ECS).
    *   Do not mount application code using volumes; `COPY` it into the image.
    *   Configure persistent volumes for database data, storage files, etc.
    *   Do not expose database ports publicly unless necessary and secured.
    *   Configure Nginx/web server specifically for production (HTTPS, caching headers, security headers).
    *   Ensure PHP-FPM is configured appropriately for production loads.
