document.addEventListener('DOMContentLoaded', function() {
    // DOM Elements
    const dateInput = document.getElementById('selected_date');
    const calendarDays = document.getElementById('calendarDays');
    const currentMonthElement = document.getElementById('currentMonth');
    const selectedDateText = document.getElementById('selectedDateText');
    const submitButton = document.getElementById('submitButton');
    const quotaWarning = document.getElementById('quotaWarning');
    const timeSlotContainer = document.getElementById('timeSlotContainer'); // Added for event delegation

    // Set timezone and locale
    moment.locale('id');
    moment.tz.setDefault('Asia/Jakarta');

    // Initialize variables
    let currentDate = moment().tz('Asia/Jakarta');
    let selectedDate = dateInput.value ? moment(dateInput.value).tz('Asia/Jakarta') : null;
    const today = moment().tz('Asia/Jakarta').startOf('day');

    // Store available and unavailable dates
    let availableDates = new Set();
    let unavailableDates = new Set();
    let isLoadingCalendar = false;
    let checkingNextMonth = false;
    let isQuotaCheckScheduled = false;
    let isTimeSlotsReady = false; // Start as false, rely on event
    let isCalendarDataReady = false; // Start as false, rely on load completion
    let retryAttempt = 0;
    let initialQuotaCheckPending = false; // Flag to indicate if initial check is waiting

    // Form Validation State
    let formValidationState = {
        hasDate: !!dateInput.value, // Initialize based on pre-filled value
        hasAvailableTimeSlot: false // Initialize to false
    };

    // Cache settings
    const CACHE_EXPIRY = 60 * 15; // 15 minutes in seconds
    const CACHE_PREFIX = 'date_availability_';
    let cacheVersion = 0; // Current cache version, will be fetched from server
    
    // Check server for cache version on page load
    async function fetchCacheVersion() {
        try {
            const response = await fetch('/api/cache-version');
            if (response.ok) {
                try {
                    const data = await response.json();
                    cacheVersion = data.version || 0;
                    // Clear all caches if version changes
                    checkAndClearCachesIfVersionChanged(cacheVersion);
                } catch (jsonError) {
                    // Silently use fallback version without logging to console
                    cacheVersion = Math.floor(Date.now() / 1000); // Use current timestamp as fallback
                }
            } else {
                // Silently use fallback version without logging to console
                cacheVersion = Math.floor(Date.now() / 1000);
            }
        } catch (error) {
            // Silently use fallback version without logging to console
            cacheVersion = Math.floor(Date.now() / 1000);
            // Still clear caches on error for safety
            clearAllCaches();
        }
    }
    
    // Call this immediately
    fetchCacheVersion();
    
    // Check and clear all caches if version changed
    function checkAndClearCachesIfVersionChanged(newVersion) {
        const storedVersion = localStorage.getItem('cache_version');
        if (storedVersion !== newVersion.toString()) {
            clearAllCaches();
            localStorage.setItem('cache_version', newVersion.toString());
        }
    }
    
    // Clear all caches
    function clearAllCaches() {
        // Get all keys in localStorage
        for (let i = 0; i < localStorage.length; i++) {
            const key = localStorage.key(i);
            // Clear date availability and quota caches
            if (key && (key.startsWith(CACHE_PREFIX) || key.startsWith('quota_'))) {
                localStorage.removeItem(key);
            }
        }
    }

    // Helper function to get formatted cache key
    function getCacheKey(year, month) {
        return `${CACHE_PREFIX}${year}_${month}`;
    }

    // Helper functions for localStorage cache
    function getFromCache(key) {
        try {
            const cachedData = localStorage.getItem(key);
            if (!cachedData) return null;

            const data = JSON.parse(cachedData);

            // Check if cache is expired or version mismatch
            if (data.timestamp && 
                (Math.floor(Date.now() / 1000) - data.timestamp) < CACHE_EXPIRY && 
                data.version === cacheVersion) {
                return data.value;
            } else {
                // Cache expired or version mismatch, remove it
                localStorage.removeItem(key);
                return null;
            }
        } catch (error) {
            return null;
        }
    }

    function saveToCache(key, value) {
        try {
            const dataToCache = {
                timestamp: Math.floor(Date.now() / 1000),
                version: cacheVersion,
                value: value
            };
            localStorage.setItem(key, JSON.stringify(dataToCache));
        } catch (error) {
            // Silently fail
        }
    }

    // Function to preload the next month's data in the background
    async function preloadNextMonth() {
        const nextMonth = currentDate.clone().add(1, 'month');
        const year = nextMonth.year();
        const month = nextMonth.month() + 1; // JavaScript months are 0-indexed

        // Check if already in cache
        const cacheKey = getCacheKey(year, month);
        if (getFromCache(cacheKey)) {
            return;
        }

        try {
            const response = await fetch(`/get-date-availability/${year}/${month}`);
            const data = await response.json();

            if (data.success) {
                saveToCache(cacheKey, {
                    available_dates: data.available_dates,
                    unavailable_dates: data.unavailable_dates
                });
            }
        } catch (error) {
            // Silently fail
        }
    }

    // Add a safety timeout to ensure the calendar is always displayed
    // This prevents the calendar from being stuck in a loading state
    function setupCalendarSafetyTimeout() {
        setTimeout(() => {
            const loadingIndicator = document.getElementById('calendar-loading');
            if (loadingIndicator) {
                loadingIndicator.remove();
                isLoadingCalendar = false;

                // Trigger an immediate retry to load data if it failed
                if (availableDates.size === 0 && unavailableDates.size === 0) {
                    setTimeout(() => {
                        loadMonthAvailability(true);
                    }, 500);
                } else {
                    updateCalendar();
                }
            }
        }, 2000); // 2 second timeout (reduced from original for faster response)
    }

    function updateCalendar() {
        const year = currentDate.year();
        const month = currentDate.month(); // JavaScript months are 0-indexed (0-11)

        // Update the month/year display at the top of the calendar
        currentMonthElement.textContent = currentDate.format('MMMM YYYY');
        calendarDays.innerHTML = '';

        const firstDay = moment([year, month, 1]).tz('Asia/Jakarta');
        const daysInMonth = firstDay.daysInMonth();
        const startingDay = firstDay.day();

        // Add empty cells for padding
        for (let i = 0; i < startingDay; i++) {
            const emptyCell = document.createElement('div');
            emptyCell.classList.add('calendar-day', 'empty');
            calendarDays.appendChild(emptyCell);
        }

        // Add days
        for (let day = 1; day <= daysInMonth; day++) {
            const date = moment([year, month, day]).tz('Asia/Jakarta');
            const dayElement = document.createElement('div');
            dayElement.classList.add('calendar-day');
            dayElement.textContent = day;

            const dateString = date.format('YYYY-MM-DD');

            // Check if date is disabled (weekend, past date, or known to be unavailable)
            if (date.isBefore(today, 'day') || date.day() === 0 || date.day() === 6) {
                dayElement.classList.add('disabled');
            } else if (unavailableDates.has(dateString)) {
                dayElement.classList.add('disabled');
                dayElement.classList.add('full-quota');
                dayElement.setAttribute('title', 'Kuota penuh untuk semua sesi');
            } else if (availableDates.has(dateString)) {
                // Date is known to be available
                dayElement.classList.add('available');
                dayElement.setAttribute('title', 'Tersedia sesi untuk tanggal ini');
                dayElement.addEventListener('click', () => selectDate(date));
            } else {
                // Date availability unknown (shouldn't happen with server-side loading)
                dayElement.classList.add('unknown');
                dayElement.setAttribute('title', 'Memuat ketersediaan...');
            }

            if (selectedDate && date.isSame(selectedDate, 'day')) {
                dayElement.classList.add('selected');
            }

            if (date.isSame(today, 'day')) {
                dayElement.classList.add('today');
            }

            calendarDays.appendChild(dayElement);
        }

        // After rendering current month, preload next month's data
        setTimeout(() => preloadNextMonth(), 1000);
    }

    function selectDate(date) {
        const jakartaDate = moment(date).tz('Asia/Jakarta');
        const dateStr = jakartaDate.format('YYYY-MM-DD');

        if (jakartaDate.day() === 0 || jakartaDate.day() === 6) {
            alert('Mohon maaf, pendaftaran hanya tersedia untuk hari kerja (Senin-Jumat)');
            return;
        }

        // Check if date is unavailable
        if (unavailableDates.has(dateStr)) {
            alert('Mohon maaf, semua sesi pada tanggal ini sudah penuh. Silakan pilih tanggal lain.');
            return;
        }

        selectedDate = jakartaDate;
        dateInput.value = dateStr;
        selectedDateText.textContent = jakartaDate.format('dddd, D MMMM YYYY');
        updateCalendar();

        // Trigger a custom event for quota-manager.js
        const event = new CustomEvent('dateSelected', {
            detail: { date: dateStr }
        });
        document.dispatchEvent(event);

        // Update form state
        formValidationState.hasDate = true;
        // Reset time slot selection when date changes
        formValidationState.hasAvailableTimeSlot = false;
        clearTimeSelection(); // Function to uncheck radio buttons
        updateSubmitButtonState(); // Update button state immediately

        // Run the quota check
        checkQuotaInternal();
    }

    // Function to clear selected time slot radio button
    function clearTimeSelection() {
        const checkedTimeInput = document.querySelector('input[name="visit_time"]:checked');
        if (checkedTimeInput) {
            checkedTimeInput.checked = false;
        }
    }

    // Function to update submit button state
    function updateSubmitButtonState() {
        const isFormValid = formValidationState.hasDate && formValidationState.hasAvailableTimeSlot;
        if (submitButton) {
            submitButton.disabled = !isFormValid;
        }
        if (quotaWarning) {
            quotaWarning.style.display = isFormValid ? 'none' : 'block';
        }
    }

    // Internal function to check quota, uses readiness flags
    function checkQuotaInternal() {
        const currentSelectedDate = dateInput.value;
        if (!currentSelectedDate) {
            isQuotaCheckScheduled = false; // Reset flag if no date
            return;
        }

        // Prerequisites check: Ensure calendar data is loaded AND time slots are ready.
        if (!isCalendarDataReady || !isTimeSlotsReady) {
            initialQuotaCheckPending = true; // Mark that a check is pending
            // No automatic retry timeout here. Will be triggered when prerequisites change.
            return; // Exit function until prerequisites are met
        }
        initialQuotaCheckPending = false; // Reset pending flag as we are proceeding

        // Reset flag as we are proceeding
        isQuotaCheckScheduled = false;

        // Get all time slot inputs (re-query in case they were added after initial load)
        const timeInputs = document.querySelectorAll('input[name="visit_time"]');
        if (timeInputs.length === 0) {
            return; // No slots to check
        }

        // Cache key for time slots
        const quotaCacheKey = `quota_${currentSelectedDate}`;
        const cachedQuotaData = getFromCache(quotaCacheKey);
        // If we have cached quota data, use it
        if (cachedQuotaData) {
            updateTimeSlotsFromCache(cachedQuotaData);
            return;
        }

        // Create an object to store quota data for caching
        const quotaDataCache = {};
        let completedRequests = 0;
        const totalRequests = timeInputs.length; // Store total count

        // Set loading state for all time slots
        timeInputs.forEach(input => {
            const selectedTime = input.value;
            const label = document.querySelector(`label[for="${input.id}"]`);

            if (!label) {
                completedRequests++; // Count as completed to avoid blocking cache save
                if (completedRequests === totalRequests && totalRequests > 0) { // Check if all are done
                    saveToCache(quotaCacheKey, quotaDataCache);
                }
                return;
            }

            const quotaInfo = label.querySelector('.quota-info');
            if (!quotaInfo) {
                completedRequests++; // Count as completed
                if (completedRequests === totalRequests && totalRequests > 0) { // Check if all are done
                    saveToCache(quotaCacheKey, quotaDataCache);
                }
                return;
            }

            const quotaText = quotaInfo.querySelector('.quota-text');
            const quotaAvailable = quotaInfo.querySelector('.quota-available');
            const quotaTotal = quotaInfo.querySelector('.quota-total');
            const quotaBadge = quotaInfo.querySelector('.quota-badge');

            // Set loading state
            if (quotaText) quotaText.textContent = 'Memeriksa kuota...';
            if (quotaAvailable) quotaAvailable.textContent = '?';
            if (quotaTotal) quotaTotal.textContent = '?';
            if (quotaBadge) quotaBadge.className = 'quota-badge badge bg-secondary';
            
            fetch(`/check-quota/${currentSelectedDate}/${selectedTime}`)
                .then(response => {
                    if (!response.ok) { // Check for HTTP errors
                        throw new Error(`HTTP error! status: ${response.status}`);
                    }
                    return response.json();
                })
                .then(data => {
                    // Store data for caching
                    quotaDataCache[selectedTime] = data;

                    updateTimeSlot(input, data);

                    // Check if all requests are done
                    completedRequests++;
                    if (completedRequests === totalRequests && totalRequests > 0) { // Check if all are done
                        saveToCache(quotaCacheKey, quotaDataCache);
                    }
                })
                .catch(error => {
                    if (quotaText) {
                        quotaText.textContent = 'Gagal memuat kuota';
                        quotaText.className = 'quota-text text-danger';
                    }
                    // Ensure the slot appears disabled on error
                    input.disabled = true;
                    if (label) label.classList.add('disabled-time-slot');
                    if (quotaBadge) quotaBadge.className = 'quota-badge badge bg-danger';
                    if (quotaAvailable) quotaAvailable.textContent = 'X';
                    if (quotaTotal) quotaTotal.textContent = 'X';

                    // Still count as completed
                    completedRequests++;
                    if (completedRequests === totalRequests && totalRequests > 0) { // Check if all are done
                        // We don't cache on error
                    }
                });
        });
    }

    // Helper function to update time slots from cached data
    function updateTimeSlotsFromCache(cachedData) {
        const timeInputs = document.querySelectorAll('input[name="visit_time"]');

        timeInputs.forEach(input => {
            const selectedTime = input.value;
            const slotData = cachedData[selectedTime];

            if (!slotData) {
                return;
            }

            updateTimeSlot(input, slotData);
        });
    }

    // Helper function to update a single time slot
    function updateTimeSlot(input, data) {
        const selectedTime = input.value;
        const label = document.querySelector(`label[for="${input.id}"]`);

        if (!label) {
            return;
        }

        const quotaInfo = label.querySelector('.quota-info');
        if (!quotaInfo) {
            return;
        }

        const quotaText = quotaInfo.querySelector('.quota-text');
        const quotaAvailable = quotaInfo.querySelector('.quota-available');
        const quotaTotal = quotaInfo.querySelector('.quota-total');
        const quotaBadge = quotaInfo.querySelector('.quota-badge');

        if (data.disabled) {
            // Slot is disabled (quota is 0)
            input.disabled = true;
            label.classList.add('disabled-time-slot');
            if (quotaText) {
                quotaText.textContent = 'Tidak tersedia';
                quotaText.className = 'quota-text text-danger fw-bold';
            }
            if (quotaBadge) quotaBadge.className = 'quota-badge badge bg-danger';
            if (quotaAvailable) quotaAvailable.textContent = '0';
            if (quotaTotal) quotaTotal.textContent = '0';
        } else if (!data.available) {
            // Slot is full
            input.disabled = true;
            label.classList.add('disabled-time-slot');
            if (quotaText) {
                quotaText.textContent = 'Kuota Penuh';
                quotaText.className = 'quota-text text-danger fw-bold';
            }
            if (quotaBadge) quotaBadge.className = 'quota-badge badge bg-danger';
            if (quotaAvailable) quotaAvailable.textContent = '0';
            if (quotaTotal) quotaTotal.textContent = data.quota;
        } else {
            // Slot is available
            input.disabled = false;
            label.classList.remove('disabled-time-slot');
            const available = data.quota - data.used;
            if (quotaText) {
                quotaText.textContent = 'Kuota Tersedia';
                quotaText.className = 'quota-text text-success';
            }
            if (quotaBadge) quotaBadge.className = 'quota-badge badge bg-success';
            if (quotaAvailable) quotaAvailable.textContent = available;
            if (quotaTotal) quotaTotal.textContent = data.quota;
        }

        // Trigger a change event in case this is the selected radio button
        if (input.checked) {
            const event = new Event('change');
            input.dispatchEvent(event);
        }
    }

    // Function to fetch and load date availability from the server
    async function loadMonthAvailability(checkNextMonthIfNeeded = true) {
        if (isLoadingCalendar) return;

        isLoadingCalendar = true;
        retryAttempt = 0;

        // Get the current month and year
        const year = currentDate.year();
        const month = currentDate.month() + 1; // JavaScript months are 0-indexed
        const cacheKey = getCacheKey(year, month);

        // First update the calendar with empty data to prevent blank state
        updateCalendar();

        // Add loading indicator to the calendar
        const loadingIndicator = document.createElement('div');
        loadingIndicator.id = 'calendar-loading';
        loadingIndicator.innerHTML = '<div class="spinner-border text-primary" role="status"><span class="visually-hidden">Loading...</span></div><div class="mt-2">Memuat ketersediaan tanggal...</div>';
        loadingIndicator.style.cssText = 'position: absolute; top: 0; left: 0; right: 0; bottom: 0; background: rgba(255,255,255,0.7); display: flex; flex-direction: column; align-items: center; justify-content: center; z-index: 100;';
        const calendarContainer = document.querySelector('.calendar-container');
        if (calendarContainer) {
            calendarContainer.style.position = 'relative';
            calendarContainer.appendChild(loadingIndicator);
        }

        // Setup safety timeout to prevent stuck loading
        setupCalendarSafetyTimeout();

        try {
            // Try to get data from cache first
            const cachedData = getFromCache(cacheKey);

            if (cachedData) {
                // Use cached data
                availableDates = new Set(cachedData.available_dates);
                unavailableDates = new Set(cachedData.unavailable_dates);

                // Update the calendar with the cached data
                updateCalendar();

                // If there are no available dates in this month and we should check next month
                if (cachedData.available_dates.length === 0 && checkNextMonthIfNeeded && !checkingNextMonth) {
                    // Always remove loading indicator before moving to next month
                    const loadingIndicator = document.getElementById('calendar-loading');
                    if (loadingIndicator) {
                        loadingIndicator.remove();
                    }

                     // Advance to next month
                     goToNextAvailableMonth();
                     return; // Stop execution here, as goToNextAvailableMonth will handle loading
                 }
             } else {
                 // Fetch fresh data from server with retry mechanism
                await fetchMonthAvailability(year, month, checkNextMonthIfNeeded);
            }
        } catch (error) {
            // Even if there's an error, update the calendar with whatever data we have
            updateCalendar();
        } finally {
            // Always remove loading indicator and reset loading state
            const loadingIndicator = document.getElementById('calendar-loading');
            if (loadingIndicator) {
                loadingIndicator.remove();
             }
             isLoadingCalendar = false;
             // Only finalize readiness and check quota if we are NOT currently switching months
             if (!checkingNextMonth) {
                 isCalendarDataReady = true; // Mark calendar data as ready ONLY when not switching months
                 // Trigger quota check if pending or date selected
                 if (initialQuotaCheckPending || dateInput.value) {
                     checkQuotaInternal();
                 }
             }
         }
     }

    // Helper function to fetch month availability with retry mechanism
    async function fetchMonthAvailability(year, month, checkNextMonthIfNeeded, maxRetries = 3) {
        const cacheKey = getCacheKey(year, month);

        try {
            const response = await fetch(`/get-date-availability/${year}/${month}`);
            const data = await response.json();

            if (data.success) {
                // Clear existing sets
                availableDates = new Set();
                unavailableDates = new Set();

                // Add available dates to the set
                data.available_dates.forEach(dateStr => {
                    availableDates.add(dateStr);
                });

                // Add unavailable dates to the set
                data.unavailable_dates.forEach(dateStr => {
                    unavailableDates.add(dateStr);
                });

                // Save valid data to cache
                saveToCache(cacheKey, {
                    available_dates: data.available_dates,
                    unavailable_dates: data.unavailable_dates
                });

                // If there are no available dates in this month and we should check next month
                if (data.available_dates.length === 0 && checkNextMonthIfNeeded && !checkingNextMonth) {
                    // Always remove loading indicator before moving to next month
                    const loadingIndicator = document.getElementById('calendar-loading');
                    if (loadingIndicator) {
                        loadingIndicator.remove();
                    }

                     // Advance to next month
                     goToNextAvailableMonth();
                     return; // Stop execution here, as goToNextAvailableMonth will handle loading
                 }

                  // Update calendar with the loaded data
                 updateCalendar();
                 // DO NOT set isCalendarDataReady here. It's handled in the finally block of loadMonthAvailability.
             } else {
                 throw new Error(data.message || 'Failed to load date availability');
             }
        } catch (error) {
            // Retry if we haven't reached max retries
            retryAttempt++;
            if (retryAttempt <= maxRetries) {
                await new Promise(resolve => setTimeout(resolve, 1000)); // Wait 1 second before retry
                return fetchMonthAvailability(year, month, checkNextMonthIfNeeded, maxRetries);
             } else {
                 // Update calendar even if we failed to get data
                 updateCalendar();
                 // isCalendarDataReady will be set in the finally block
             }
         }
    }

    // This function will check and advance to the next month with available slots
    async function goToNextAvailableMonth(monthsToCheck = 3) {
        if (checkingNextMonth) return;
        checkingNextMonth = true;

        try {
            let foundAvailableMonth = false; // Flag to track if we found one
            for (let i = 0; i < monthsToCheck; i++) {
                // Move to next month
                currentDate = currentDate.clone().add(1, 'month');
                const year = currentDate.year();
                const month = currentDate.month() + 1; // JavaScript months are 0-indexed

                // Try to get data from cache first
                const cacheKey = getCacheKey(year, month);
                let cachedData = getFromCache(cacheKey);

                // If not in cache, fetch it
                if (!cachedData) {
                    try {
                        const response = await fetch(`/get-date-availability/${year}/${month}`);
                        const data = await response.json();

                        if (data.success) {
                            cachedData = {
                                available_dates: data.available_dates,
                                unavailable_dates: data.unavailable_dates
                            };

                            saveToCache(cacheKey, cachedData);
                        }
                    } catch (error) {
                        continue; // Skip to next month if this one fails
                    }
                }

                // If we have available dates in this month, load it and stop
                if (cachedData && cachedData.available_dates && cachedData.available_dates.length > 0) {
                    // Update UI to show this month first
                    updateCalendar();
                    // Reset flags before calling loadMonthAvailability again
                    checkingNextMonth = false;
                    isLoadingCalendar = false;
                    // Load this month's data properly, setting isCalendarDataReady in its finally block
                    loadMonthAvailability(false);
                    return; // Exit the loop and function
                }
            } // End for loop

            // If we finished the loop and didn't find an available month
            // Load the last month checked (which is already set in currentDate)
            updateCalendar(); // Update UI to show this month first
            // Reset flags before calling loadMonthAvailability again
            checkingNextMonth = false;
            isLoadingCalendar = false;
            // Load this month's data properly, setting isCalendarDataReady in its finally block
            loadMonthAvailability(false);
        } catch (error) {
            // Ensure flags are reset even on error
            checkingNextMonth = false;
            isLoadingCalendar = false;
            // Attempt to load the current month as a fallback
            loadMonthAvailability(false);
        }
    }

    // Event Listeners
    document.getElementById('prevMonth').addEventListener('click', () => {
        currentDate = currentDate.clone().subtract(1, 'month');
        // Load availability data for the new month without auto-advancing
        loadMonthAvailability(false);
    });

    document.getElementById('nextMonth').addEventListener('click', () => {
        currentDate = currentDate.clone().add(1, 'month');
        // Load availability data for the new month without auto-advancing
        loadMonthAvailability(false);
    });

    // Initialize calendar (with a short delay to make sure DOM is ready)
    setTimeout(() => {
        // First render calendar grid
        updateCalendar();

        // Then load data with a slight delay
        setTimeout(() => {
            loadMonthAvailability(true);
        }, 200);
    }, 100);

    // If a date was pre-selected (e.g., from form validation error), ensure selectedDate is set
    if (dateInput.value) {
        selectedDate = moment(dateInput.value).tz('Asia/Jakarta');
    }

    // Add CSS for calendar styling
    const style = document.createElement('style');
    style.textContent = `
        .calendar-day.full-quota {
            background-color: #f8f9fa;
            color: #6c757d;
            position: relative;
            cursor: not-allowed !important;
        }

        .calendar-day.available {
            background-color: white;
            color: #212529;
            position: relative;
            cursor: pointer !important;
        }

        .calendar-day.unknown {
            background-color: #f8f9fa;
            color: #6c757d;
            cursor: wait !important;
        }

        .calendar-container {
            position: relative;
            min-height: 300px;
        }

        /* Add refresh button */
        .calendar-header {
            position: relative;
        }

        #refreshCalendar {
            position: absolute;
            right: 10px;
            top: 50%;
            transform: translateY(-50%);
            font-size: 1rem;
            cursor: pointer;
            color: #0d6efd;
        }
    `;
    document.head.appendChild(style);

    // Add refresh button to clear cache and reload
    const calendarHeader = document.querySelector('.calendar-header');
    if (calendarHeader) {
        const refreshButton = document.createElement('span');
        refreshButton.id = 'refreshCalendar';
        refreshButton.innerHTML = '<i class="bi bi-arrow-clockwise"></i>';
        refreshButton.title = 'Muat ulang ketersediaan tanggal';
        refreshButton.addEventListener('click', function() {
            // Clear cache for current month
            const year = currentDate.year();
            const month = currentDate.month() + 1;
            const cacheKey = getCacheKey(year, month);
            localStorage.removeItem(cacheKey);

            // Reload data
            loadMonthAvailability(false);
        });
        calendarHeader.appendChild(refreshButton);
    }

    // Listen for the custom event from Blade when time slots are rendered
    document.addEventListener('timeSlotsRendered', function() {
        isTimeSlotsReady = true; // Mark time slots as ready
        // Trigger quota check if it was pending due to slots not being ready
        if (initialQuotaCheckPending) {
            checkQuotaInternal(); // Trigger the internal check now that slots are ready
        }
    });

    // Initial check for submit button state based on pre-filled values
    // Check if a time was pre-selected and is valid
    const preSelectedTimeInput = document.querySelector('input[name="visit_time"]:checked');
    if (preSelectedTimeInput && !preSelectedTimeInput.disabled) {
        formValidationState.hasAvailableTimeSlot = true;
    }
    updateSubmitButtonState(); // Update button state on initial load

    // Add event listener for time slot changes using event delegation
    if (timeSlotContainer) {
        timeSlotContainer.addEventListener('change', function(e) {
            if (e.target && e.target.matches('input[name="visit_time"]')) {
                const selectedRadio = e.target;
                formValidationState.hasAvailableTimeSlot = selectedRadio.checked && !selectedRadio.disabled;
                updateSubmitButtonState();
            }
        });
    }
});
