<?php
// routes/web.php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\RegistrationController;
use App\Http\Controllers\AdminController;
use App\Http\Controllers\ImportController;
use App\Http\Controllers\SettingController;
use App\Http\Controllers\CacheController;

// Home Route - Welcome page as the default homepage
Route::get('/', function() {
    return view('welcome');
})->name('home');

// Public Routes
Route::get('/terms', [RegistrationController::class, 'terms'])->name('terms');
Route::get('/register', [RegistrationController::class, 'showForm'])->name('register.form');
Route::post('/register', [RegistrationController::class, 'store'])->name('register.store');
Route::get('/check-quota/{date}/{time}', [RegistrationController::class, 'checkQuota'])
    ->name('check.quota')
    ->where([
        'date' => '[0-9]{4}-[0-9]{2}-[0-9]{2}',
        'time' => '[0-9]{2}:[0-9]{2}'
    ]);
    
// Batch quota checking endpoint for optimized performance
Route::get('/check-quotas/{date}', [RegistrationController::class, 'checkQuotaBatch'])
    ->name('check.quota.batch')
    ->where([
        'date' => '[0-9]{4}-[0-9]{2}-[0-9]{2}'
    ]);
    
Route::get('/get-time-slots', [RegistrationController::class, 'getActiveTimeSlots'])
    ->name('get.time-slots');
Route::get('/booking-print/{id}', [RegistrationController::class, 'printVersion'])->name('booking.print');
Route::get('/get-date-availability/{year?}/{month?}', [RegistrationController::class, 'getDateAvailability'])
    ->name('get.date-availability')
    ->where([
        'year' => '[0-9]{4}',
        'month' => '[0-9]{1,2}'
    ]);

// Cache version API endpoint for client-side caching
Route::get('/api/cache-version', [CacheController::class, 'getCacheVersion']);

// Admin Routes
Route::get('/admin', [AdminController::class, 'loginForm'])->name('admin.login');
Route::get('/admin/login', [AdminController::class, 'loginForm'])->name('admin.login.form'); // Nama route diubah
Route::post('/admin/login', [AdminController::class, 'login'])->name('admin.login.post');
Route::post('/admin/logout', [AdminController::class, 'logout'])->name('admin.logout');

// Protected Admin Routes
Route::middleware(['auth:admin'])->name('admin.')->prefix('admin')->group(function () {
    Route::get('/dashboard', [AdminController::class, 'dashboard'])->name('dashboard');
    Route::get('/statistics/load', [AdminController::class, 'loadStatistics'])->name('statistics.load');
    Route::get('/registrations', [AdminController::class, 'registrations'])->name('registrations');
    
    // Import Routes
    Route::get('/import', [ImportController::class, 'showImportForm'])->name('import');
    Route::post('/import/process', [ImportController::class, 'processImport'])->name('import.process');
    
    // Settings Routes
    Route::get('/settings', [SettingController::class, 'edit'])->name('settings');
    Route::post('/settings/update', [SettingController::class, 'update'])->name('settings.update');
    
    // Time Slot Management Routes
    Route::post('/time-slots', [SettingController::class, 'storeTimeSlot'])->name('time-slots.store');
    Route::put('/time-slots/{id}', [SettingController::class, 'updateTimeSlot'])->name('time-slots.update');
    Route::delete('/time-slots/{id}', [SettingController::class, 'deleteTimeSlot'])->name('time-slots.delete');
    Route::post('/time-slots/reorder', [SettingController::class, 'reorderTimeSlots'])->name('time-slots.reorder');
    
    // Cache management route for admin
    Route::post('/cache/invalidate', [CacheController::class, 'incrementCacheVersion'])->name('cache.invalidate');
    
    Route::get('/reports', [AdminController::class, 'reports'])->name('reports');
    Route::post('/reports/export', [AdminController::class, 'exportReport'])->name('reports.export');
    Route::post('/download/ktp/zip', [AdminController::class, 'downloadKtpZip'])->name('download.ktp.zip');
    Route::post('/holiday', [AdminController::class, 'storeHoliday'])->name('holiday.store');
    Route::delete('/holiday/{id}', [AdminController::class, 'deleteHoliday'])->name('holiday.delete');
    
    // KTP Image Routes
    Route::get('/ktp/{registration}/view', [AdminController::class, 'serveKtpImage'])
        ->name('serve.ktp');
    Route::get('/ktp/{registration}/download', [AdminController::class, 'downloadKtp'])
        ->name('download.ktp');
    Route::get('/transaction/ktp/{id}/view', [AdminController::class, 'serveTransactionKtpImage'])
        ->name('serve.transaction.ktp');
    Route::get('/transaction/ktp/{id}/download', [AdminController::class, 'downloadTransactionKtp'])
        ->name('download.transaction.ktp');
});

// Fallback route for any URL that doesn't match the above routes
// This will redirect users to home page instead of showing 404 error
// It also handles expired sessions by redirecting to homepage
Route::fallback(function() {
    return redirect()->route('home');
});
