<?php

namespace Tests\Feature;

use App\Models\Admin;
use App\Models\Registration;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;

class AdminTest extends TestCase
{
    use RefreshDatabase, WithFaker;

    private Admin $admin;

    protected function setUp(): void
    {
        parent::setUp();
        $this->admin = Admin::factory()->create();
    }

    /**
     * Test login admin berhasil
     */
    public function test_admin_can_login(): void
    {
        $response = $this->post('/admin/login', [
            'username' => $this->admin->username,
            'password' => 'password'
        ]);

        $response->assertStatus(302);
        $response->assertRedirect('/admin/dashboard');
        $this->assertAuthenticated('admin');
    }

    /**
     * Test login gagal dengan kredensial salah
     */
    public function test_admin_cannot_login_with_invalid_credentials(): void
    {
        $response = $this->post('/admin/login', [
            'username' => $this->admin->username,
            'password' => 'wrong-password'
        ]);

        $response->assertStatus(302);
        $response->assertSessionHasErrors();
        $this->assertGuest('admin');
    }

    /**
     * Test admin dapat melihat daftar registrasi
     */
    public function test_admin_can_view_registrations(): void
    {
        // Login sebagai admin
        $this->actingAs($this->admin, 'admin');

        // Buat beberapa registrasi dummy
        Registration::factory()->count(5)->create();

        $response = $this->get('/admin/dashboard');

        $response->assertStatus(200);
        $response->assertViewHas('registrations');
    }

    /**
     * Test admin dapat memperbarui status registrasi
     */
    public function test_admin_can_update_registration_status(): void
    {
        $this->actingAs($this->admin, 'admin');

        $registration = Registration::factory()->create([
            'status' => 'pending'
        ]);

        $response = $this->post("/admin/registrations/{$registration->id}/status", [
            'status' => 'approved'
        ]);

        $response->assertStatus(302);
        $this->assertDatabaseHas('registrations', [
            'id' => $registration->id,
            'status' => 'approved'
        ]);
    }

    /**
     * Test admin dapat mengunduh laporan
     */
    public function test_admin_can_download_report(): void
    {
        $this->actingAs($this->admin, 'admin');

        // Buat beberapa registrasi untuk laporan
        Registration::factory()->count(3)->create();

        $response = $this->get('/admin/reports/export', [
            'start_date' => now()->subDays(7)->format('Y-m-d'),
            'end_date' => now()->format('Y-m-d')
        ]);

        $response->assertStatus(200);
        $response->assertHeader('content-type', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    }

    /**
     * Test akses halaman admin terproteksi
     */
    public function test_guest_cannot_access_admin_pages(): void
    {
        $response = $this->get('/admin/dashboard');
        $response->assertStatus(302);
        $response->assertRedirect('/admin/login');
    }

    /**
     * Test admin dapat mengatur kuota
     */
    public function test_admin_can_manage_quota(): void
    {
        $this->actingAs($this->admin, 'admin');

        $response = $this->post('/admin/settings/quota', [
            'date' => now()->addDays(1)->format('Y-m-d'),
            'quota' => 50
        ]);

        $response->assertStatus(302);
        $this->assertDatabaseHas('quotas', [
            'date' => now()->addDays(1)->format('Y-m-d'),
            'quota' => 50
        ]);
    }
}
