<?php

namespace Tests\Feature;

use App\Models\Registration;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;

class RegistrationTest extends TestCase
{
    use RefreshDatabase, WithFaker;

    /**
     * Test berhasil membuat registrasi baru
     */
    public function test_can_create_registration(): void
    {
        $registrationData = [
            'name' => 'Test User',
            'nik' => '1234567890123456',
            'kk_number' => '9876543210123456',
            'gender' => 'Laki-laki',
            'phone' => '081234567890',
            'email' => 'test@example.com',
            'address' => 'Jl. Test No. 123',
            'registration_date' => now()->addDays(1)->format('Y-m-d')
        ];

        $response = $this->post('/register', $registrationData);

        $response->assertStatus(302); // Redirect setelah sukses
        $this->assertDatabaseHas('registrations', [
            'name' => 'Test User',
            'nik' => '1234567890123456'
        ]);
    }

    /**
     * Test validasi data registrasi
     */
    public function test_registration_requires_valid_data(): void
    {
        $response = $this->post('/register', [
            'name' => '', // Kosong - seharusnya error
            'nik' => 'invalid-nik', // Format salah
            'registration_date' => now()->subDay()->format('Y-m-d') // Tanggal lampau - seharusnya error
        ]);

        $response->assertSessionHasErrors(['name', 'nik', 'registration_date']);
    }

    /**
     * Test duplikasi NIK
     */
    public function test_cannot_register_with_duplicate_nik(): void
    {
        // Buat registrasi pertama
        $existingRegistration = Registration::factory()->create();

        // Coba buat registrasi dengan NIK yang sama
        $response = $this->post('/register', [
            'name' => 'Another User',
            'nik' => $existingRegistration->nik,
            'kk_number' => '8765432109876543',
            'gender' => 'Perempuan',
            'phone' => '089876543210',
            'email' => 'another@example.com',
            'address' => 'Jl. Another No. 456',
            'registration_date' => now()->addDays(2)->format('Y-m-d')
        ]);

        $response->assertSessionHasErrors('nik');
    }

    /**
     * Test batasan kuota harian
     */
    public function test_registration_respects_daily_quota(): void
    {
        // Set kuota harian ke 2
        $date = now()->addDay()->format('Y-m-d');
        
        // Buat 2 registrasi untuk memenuhi kuota
        Registration::factory()->count(2)->create([
            'registration_date' => $date
        ]);

        // Coba buat registrasi ketiga
        $response = $this->post('/register', [
            'name' => 'Quota Test User',
            'nik' => '5432109876543210',
            'kk_number' => '6543210987654321',
            'gender' => 'Laki-laki',
            'phone' => '087654321098',
            'email' => 'quota@example.com',
            'address' => 'Jl. Quota No. 789',
            'registration_date' => $date
        ]);

        $response->assertSessionHasErrors(); // Seharusnya error karena kuota penuh
    }
}
