<?php

namespace Tests\Unit;

use Tests\TestCase;
use App\Models\Registration;
use App\Models\Setting;
use App\Models\Holiday;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;

class RegistrationTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        Storage::fake('private');
    }

    public function test_registration_form_validation()
    {
        $response = $this->post('/register', []);
        $response->assertSessionHasErrors(['name', 'nik', 'phone', 'visit_date', 'visit_time', 'ktp_image']);
    }

    public function test_can_upload_ktp_image()
    {
        $file = UploadedFile::fake()->image('ktp.jpg');
        
        $response = $this->post('/register', [
            'name' => 'Test User',
            'nik' => '1234567890123456',
            'phone' => '6281234567890',
            'email' => 'test@example.com',
            'visit_date' => Carbon::tomorrow()->format('Y-m-d'),
            'visit_time' => '08:00',
            'ktp_image' => $file
        ]);

        Storage::disk('private')->assertExists('ktp_images/' . date('Y/m') . '/' . $file->hashName());
    }

    public function test_cannot_book_on_holiday()
    {
        $holiday = Holiday::create([
            'date' => Carbon::tomorrow(),
            'description' => 'Test Holiday'
        ]);

        $response = $this->get('/check-quota/' . Carbon::tomorrow()->format('Y-m-d') . '/08:00');
        $response->assertJson(['available' => false, 'message' => 'Hari Libur']);
    }

    public function test_cannot_exceed_quota()
    {
        Setting::create([
            'key' => 'quota_08_00',
            'value' => '1'
        ]);

        // Create one registration
        Registration::create([
            'booking_code' => 'TEST001',
            'name' => 'Test User 1',
            'nik' => '1234567890123456',
            'phone' => '6281234567890',
            'visit_date' => Carbon::tomorrow()->format('Y-m-d'),
            'visit_time' => '08:00'
        ]);

        $response = $this->get('/check-quota/' . Carbon::tomorrow()->format('Y-m-d') . '/08:00');
        $response->assertJson(['available' => false]);
    }

    public function test_successful_registration()
    {
        Setting::create([
            'key' => 'quota_08_00',
            'value' => '15'
        ]);

        $file = UploadedFile::fake()->image('ktp.jpg');
        
        $response = $this->post('/register', [
            'name' => 'Test User',
            'nik' => '1234567890123456',
            'phone' => '6281234567890',
            'email' => 'test@example.com',
            'topic' => 'Test Topic',
            'priority' => true,
            'pendamping' => 'Test Pendamping',
            'visit_date' => Carbon::tomorrow()->format('Y-m-d'),
            'visit_time' => '08:00',
            'ktp_image' => $file
        ]);

        $this->assertDatabaseHas('registrations', [
            'name' => 'Test User',
            'nik' => '1234567890123456',
            'topic' => 'Test Topic',
            'priority' => true,
            'pendamping' => 'Test Pendamping'
        ]);
    }
}